{ NetCalc - IP unit 13.02

  Copyright (C) 2013 Maciej Kaczkowski / maciej@keit.co

  This library is free software; you can redistribute it and/or modify it
  under the terms of the GNU Library General Public License as published by
  the Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
  for more details.

  You should have received a copy of the GNU Library General Public License
  along with this library; if not, write to the Free Software Foundation,
  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}
unit ncstruct;

{$mode delphi}
{$H+}

interface

uses
  Classes, SysUtils;

type

  { TIPv4 }

  TIPv4BlockType = (i4btNormal, i4btPrivate, i4btLoopback, i4btLinkLocal,
    i4btMulticast, i4bt6to4, i4btSAS);

  TIPv4 = record
  private
    FRaw: DWord;
    function GetAsBin: string;
    function GetAsHex: string;
    function GetAsString: string;
    function GetOctet(Index: Byte): Byte;
    procedure SetAsString(AValue: string);
  public
    procedure Clear;

    property AsString: string read GetAsString write SetAsString;
    property AsDWord: DWord read FRaw write FRaw;
    property AsHex: string read GetAsHex;
    property AsBin: string read GetAsBin;

    property Octet[Index: Byte]: Byte read GetOctet;

    class operator Equal(a, b: TIPv4): Boolean;
    class operator NotEqual(a, b: TIPv4): Boolean;

    class operator GreaterThan(a, b: TIPv4): Boolean;
    class operator GreaterThanOrEqual(a, b: TIPv4): Boolean;
    class operator LessThan(a, b: TIPv4): Boolean;
    class operator LessThanOrEqual(a, b: TIPv4): Boolean;

    class operator Inc(a: TIPv4): TIPv4;
    class operator Dec(a: TIPv4): TIPv4;
    class operator BitwiseAnd(a, b: TIPv4): TIPv4;
    class operator BitwiseOr (a, b: TIPv4): TIPv4;
    class operator BitwiseXor(a, b: TIPv4): TIPv4;
    class operator LogicalNot(a: TIPv4): TIPv4;
  end;

  { TIPv6 }

  // http://en.wikipedia.org/wiki/IPv6_address
  TIPv6BlockType = (
    i6btNormal,
    i6btLoopback, i6btLinkLocal, // Local addresses
    i6btULA, // Unique local addresses
    i6btIPv4Compatibe, i6btIPv4Mapped, i6btSIIT, i6bt6to4, i6btTeredo, // Transition from IPv4
    i6btORCHID, i6btBMWG, // Special-purpose addresses
    i6btDocumentation, // Documentation
    i6btMulticast // Multicast addresses
    );

  TIPv6 = record
  private
    FRaw: array[0..15] of Byte; // 16 * 8 = 128 bits
    function GetAsAlternativeString: string;
    function GetAsCompressedString: string;
    function GetAsString: string;
    function GetBits(Index: Byte): Byte;
    function GetBytes(Index: Byte): Byte;
    function GetWords(Index: Byte): Word;
    procedure SetAsString(AValue: string);
    procedure SetBits(Index: Byte; AValue: Byte);
    procedure SetBytes(Index: Byte; AValue: Byte);
    procedure SetWords(Index: Byte; AValue: Word);
    function GetAsRawString: string;
  public
    procedure Clear;

    property AsString: string read GetAsString write SetAsString;
    property AsCompressedString: string read GetAsCompressedString;
    property AsAlternativeString: string read GetAsAlternativeString;

    property Bits[Index: Byte]: Byte read GetBits write SetBits;    // 1..128
    property Bytes[Index: Byte]: Byte read GetBytes write SetBytes; // 0..15
    property Words[Index: Byte]: Word read GetWords write SetWords; // 0..7

    class operator Equal(a, b: TIPv6): Boolean;
    class operator NotEqual(a, b: TIPv6): Boolean;

    class operator GreaterThan(a, b: TIPv6): Boolean;
    class operator GreaterThanOrEqual(a, b: TIPv6): Boolean;
    class operator LessThan(a, b: TIPv6): Boolean;
    class operator LessThanOrEqual(a, b: TIPv6): Boolean;

    class operator Inc(a: TIPv6): TIPv6;
    class operator Dec(a: TIPv6): TIPv6;
    class operator BitwiseAnd(a, b: TIPv6): TIPv6;
    class operator BitwiseOr (a, b: TIPv6): TIPv6;
    class operator BitwiseXor(a, b: TIPv6): TIPv6;
    class operator LogicalNot(a: TIPv6): TIPv6;
  end;

// IPv4

function IPv4Network(AIP, AMask: TIPv4): TIPv4;
function IPv4Broadcast(AIP, AMask: TIPv4): TIPv4;
function IPv4Hosts(AMask: TIPv4): Int64;
function IPv4HostsAndNetwork(AMask: TIPv4): Int64;
function IPv4Subnets(AMask: TIPv4): Int64;
function IPv4Class(AIP: TIPv4): Char;
function IPv4FirstHost(AIP, AMask: TIPv4): TIPv4;
function IPv4LastHost(AIP, AMask: TIPv4): TIPv4;
function IPv4SimilarBits(AIP1, AIP2: TIPv4): Byte;

function CIDRToMask(ABits: Byte): TIPv4;
function MaskToCIDR(AMask: TIPv4): Byte;

function IsValidIPv4(AValue: string): Boolean;
function IsValidMask(AValue: string): Boolean;

function IPv4BlockType(AIP, AMask: TIPv4): TIPv4BlockType;
function IPv4BlockTypeToStr(ABlockType: TIPv4BlockType): string;

function IPv4ToStr(AValue: TIPv4): string;
function StrToIPv4(AValue: string): TIPv4;

// IPv4 -> IPv6

function IPv4toIPv6Compatible(host: TIPv4): TIPv6;
function IPv4toIPv6Mapped(host: TIPv4): TIPv6;
function IPv4toIPv66to4(host: TIPv4): TIPv6;
function IPv4toIPv66over4(host: TIPv4): TIPv6;
function IPv4toIPv6Teredo(host, server: TIPv4; port: Word): TIPv6;
function IPv4toIPv6SIIT(host: TIPv4): TIPv6;

// IPv6

function IPv6Expand(s: string): string;
function IsValidIPv6(s: string): Boolean;

function IPv6NetworkStart(AIP: TIPv6; AMask: Byte): TIPv6;
function IPv6NetworkEnd(AIP: TIPv6; AMask: Byte): TIPv6;

function IPv6ToStr(AValue: TIPv6): string;
function StrToIPv6(AValue: string): TIPv6;

function IPv6BlockType(AIP: TIPv6; AMask: Byte): TIPv6BlockType;
function IPv6BlockTypeToStr(ABlockType: TIPv6BlockType): string;

implementation

uses
  StrUtils;

resourcestring
  RS_IP4_NORMAL = 'Unicast';
  RS_IP4_BT_PRIVATE_ADDRESS = 'Private address';
  RS_IP4_BT_LOOPBACK = 'Loopback';
  RS_IP4_BT_LINK_LOCAL = 'Link Local';
  RS_IP4_BT_MULTICAST = 'Multicast';
  RS_IP4_BT_6_TO_4 = 'IPv6 to IPv4 relay';
  RS_IP4_BT_SAS = 'Shared Address Space';

  RS_IP6_NORMAL = 'Unicast';
  RS_IP6_LOOPBACK = 'Loopback';
  RS_IP6_LINK_LOCAL = 'Link Local';
  RS_IP6_IPv4 = 'IPv4 Compatible';
  RS_IP6_ULA = 'Unique local addresses';
  RS_IP6_SIIT = 'Stateless IP/ICMP Translation';
  RS_IP6_6_TO_4 = '6to4';
  RS_IP6_TEREDO = 'Teredo tunneling';
  RS_IP6_ORCHID = 'Overlay Routable Cryptographic Hash Identifiers';
  RS_IP6_BMWG = 'Benchmarking Methodology Working Group';
  RS_IP6_DOCUMENTATION = 'Documentation';
  RS_IP6_MULTICAST = 'Multicast';

function IsValidIPv6Expanded(s: string): Boolean;
var
  i: Integer;
  c: Integer;
  ts: string;
  checkIPv4: Boolean;
begin
  Result := Length(s) > 0;
  checkIPv4 := False;
  if not Result then
    Exit;

  c := 0;
  ts := '';

  for i := 1 to Length(s) do // single loop
  begin
    if (s[i] = ':') then
    begin
      Inc(c);
      ts := '';
    end
    else
    if s[i] in ['0'..'9', 'a'..'f', 'A'..'F'] then
    begin
      ts := ts + s[i];
    end
    else
    if (s[i] = '.') and (c = 6) then // IPv4 in last segment
    begin
      ts := ts + s[i];
      checkIPv4 := True;
    end
    else
    begin
      Result := False;
      Exit;
    end;

    if (c > 7) or ((Length(ts) > 4) and (not checkIPv4)) then // segment length > 4
    begin
      Result := False;
      Exit;
    end;

    if (c > 1) then
    begin
      // :: shouldn't be here, first was deleted on expanding
      if (s[i-1] = s[i]) and (s[i] = ':') then
      begin
        Result := False;
        Exit;
      end;
    end;
  end;

  if checkIPv4 then
  begin
    if (not IsValidIPv4(ts)) or (c <> 6) then
    begin
      Result := False;
      Exit;
    end;
  end
  else
  begin
    if c <> 7 then
    begin
      Result := False;
      Exit;
    end;
  end;

  if Length(s) > 0 then
  begin
    if (s[Length(s)] = ':') or (s[1] = ':') then // :1111:
    begin
      Result := False;
      Exit;
    end;
  end;
end;

function IPv4Network(AIP, AMask: TIPv4): TIPv4;
begin
  Result := AIP and AMask;
end;

function IPv4Broadcast(AIP, AMask: TIPv4): TIPv4;
begin
  Result := AIP or (not AMask);
end;

function IPv4Hosts(AMask: TIPv4): Int64;
begin
  AMask := not AMask;
  case AMask.AsDWord of
    0: Result := -1; // http://www.ietf.org/rfc/rfc1878.txt - single host
    1: Result := -2; // http://www.ietf.org/rfc/rfc3021.txt - point to point
  else
    Result := Int64(AMask.AsDWord) - 1;
  end
end;

function CIDRToMask(ABits: Byte): TIPv4;
var
  x: DWord;
begin
  x := $ffffffff;
  ABits := 32 - ABits;
  while ABits > 0 do
  begin
    x := x shl 1;
    Dec(ABits);
  end;
  Result.AsDWord := x;
end;

function MaskToCIDR(AMask: TIPv4): Byte;
var
  x, y: DWord;
begin
  Result := 0;
  if IsValidMask(AMask.AsString) then
  begin
    x := AMask.AsDWord;
    while x > 0 do
    begin
      y := x mod 2;
      if y = 1 then
        Inc(Result);
      x := x div 2;
    end;
  end;
end;

function IPv4HostsAndNetwork(AMask: TIPv4): Int64;
begin
  Result := (not AMask).AsDWord;
end;

function IPv4Subnets(AMask: TIPv4): Int64;
begin
  Result := MaskToCIDR(AMask);
  if Result < 31 then
    Result := 1 shl (30-Result)
  else
    Result := 0;
end;

function IPv4Class(AIP: TIPv4): Char;
var
  b: Byte;
  i: Byte;
begin
  // http://en.wikipedia.org/wiki/Classful_network
  b := AIP.Octet[0];
  if b shr 4 = $f then  // 1111b
    Result := 'E'
  else
  if b shr 4 = $e then  // 1110b
    Result := 'D'
  else
  if b shr 5 = $6 then  // 110b
    Result := 'C'
  else
  if b shr 6 = 2 then   // 10b
    Result := 'B'
  else
  if b shr 7 = 0 then   // 0b
    Result := 'A'
  else
    Result := '?'
end;

function IPv4FirstHost(AIP, AMask: TIPv4): TIPv4;
begin
  Result := IPv4Network(AIP, AMask);
  Inc(Result);
end;

function IPv4LastHost(AIP, AMask: TIPv4): TIPv4;
begin
  Result := IPv4Broadcast(AIP, AMask);
  Dec(Result);
end;

function ReverseBits(x: DWord): DWord;
var
  i: Integer;
begin
  Result := 0;
  for i := 0 to 31 do
  begin
    Result := Result shl 1;
    Result := Result or (x mod 2);
    x := x shr 1;
  end;
end;

function IPv4SimilarBits(AIP1, AIP2: TIPv4): Byte;
var
  x, y: DWord;
begin
  Result := 0;
  x := ReverseBits(AIP1.AsDWord);
  y := ReverseBits(AIP2.AsDWord);
  while (x mod 2) = (y mod 2) do
  begin
    x := x div 2;
    y := y div 2;
    Inc(Result);
    if Result = 32 then
      Break;
  end;
//  Result := 32-Result;
end;

function IsValidMask(AValue: string): Boolean;
var
  ip: TIPv4;
  dw: DWord;
  b: Boolean;
begin
  Result := IsValidIPv4(AValue);
  if Result then
  begin
    ip.AsString := AValue;
    dw := ip.AsDWord;
    b := False;
    while dw > 0 do
    begin
      if (dw mod 2 = 0) and b then
      begin
        Result := False;
        Break;
      end
      else
      if (dw mod 2 = 1) then
        b := True;
      dw := dw div 2;
    end;
  end;
end;

function IPv4BlockType(AIP, AMask: TIPv4): TIPv4BlockType;

  function CheckBlock(AInIP: string; AMaskPrefix, AMaskBits: Integer): Boolean;
  var
    ip, mask: TIPv4;
  begin
    ip.AsString := AInIP;
    mask := CIDRToMask(AMaskPrefix);
    Result := (IPv4Network(ip, mask) <= AIP) and (AIP <= IPv4Broadcast(ip, mask))
          and (AMaskBits >= AMaskPrefix);
  end;

var
  i: Byte;
begin
  // http://www.ietf.org/rfc/rfc1918.txt
  i := MaskToCIDR(AMask);
  if CheckBlock('10.0.0.0', 8, i) or
     CheckBlock('172.16.0.0', 12, i) or
     CheckBlock('192.168.0.0', 16, i) then
  begin
    Result := i4btPrivate;
  end
  else
  // http://www.ietf.org/rfc/rfc1700.txt
  if CheckBlock('127.0.0.0', 8, i) then
  begin
    Result := i4btLoopback;
  end
  else
  // http://www.ietf.org/rfc/rfc3171.txt
  if CheckBlock('224.0.0.0', 4, i) then
  begin
    Result := i4btMulticast;
  end
  else
  // http://www.ietf.org/rfc/rfc3330.txt
  if CheckBlock('169.254.0.0', 16, i) then
  begin
    Result := i4btLinkLocal;
  end
  else
  // http://www.ietf.org/rfc/rfc3068.txt
  if CheckBlock('192.88.99.0', 24, i) then
  begin
    Result := i4bt6to4;
  end
  else
  // http://www.ietf.org/rfc/rfc6598.txt
  if CheckBlock('100.64.0.0', 10, i) then
  begin
    Result := i4btSAS;
  end
  else
    Result := i4btNormal;
end;

function IPv4BlockTypeToStr(ABlockType: TIPv4BlockType): string;
begin
  case ABlockType of
    i4btNormal:
      Result := '';
    i4btPrivate:
      Result := RS_IP4_BT_PRIVATE_ADDRESS;
    i4btLoopback:
      Result := RS_IP4_BT_LOOPBACK;
    i4btLinkLocal:
      Result := RS_IP4_BT_LINK_LOCAL;
    i4btMulticast:
      Result := RS_IP4_BT_MULTICAST;
    i4bt6to4:
      Result := RS_IP4_BT_6_TO_4;
    i4btSAS:
      Result := RS_IP4_BT_SAS;
  end;
end;

function IPv4ToStr(AValue: TIPv4): string;
begin
  Result := AValue.AsString;
end;

function StrToIPv4(AValue: string): TIPv4;
begin
  Result.AsString := AValue;
end;

function IsValidIPv4(AValue: string): Boolean;

  function Check(tmp: string): Boolean;
  var
    j: Integer;
  begin
    j := StrToIntDef(tmp, -1);
    Result := (j >= 0) and (j <= 255);
  end;

var
  i: Integer;
  tmp: string;
  k: Integer;
begin
  Result := True;
  k := 0;
  tmp := '';
  for i := 1 to Length(AValue) do
  begin
    if AValue[i] = '.' then
    begin
      Inc(k);
      Result := Check(tmp);
      tmp := '';
    end
    else
      tmp := tmp + AValue[i];

    if (k > 3) then
      Result := False;

    if not Result then
      Break;
  end;

  Result := Result and Check(tmp) and (k = 3);
end;

{ TIPv6 }

function TIPv6.GetBytes(Index: Byte): Byte;
begin
  if (0 <= Index) and (Index <= 15) then
    Result := FRaw[Index]
  else
    Result := 0;
end;

function TIPv6.GetAsString: string;
var
  i: Integer;
begin
  for i := 0 to 15 do
  begin
    Result := Result + IntToHex(FRaw[i], 2);
    if (i mod 2 = 1) and (i < 15) then
      Result := Result + ':'
  end;
end;

function TIPv6.GetBits(Index: Byte): Byte;
begin
  if (1 <= Index) and (Index <= 128) then
  begin
    Result := FRaw[(Index-1) div 8];
    Result := (Result shr (8 - (Index mod 8))) and 1;
  end
  else
    Result := 0;
end;

function TIPv6.GetAsCompressedString: string;
var
  i: Integer;
  w: Word;
  x, y: Integer;
  lastx, lasty, lastmax: Integer;
  b: Boolean;
begin
  i := 0;

  x := -1;
  y := x;
  lastmax := x;
  lastx := x;
  lasty := x;

  for i := 0 to 7 do
  begin
    w := GetWords(i);
    if (w = 0) and (x = -1) then
    begin
      x := i;
      y := i;
    end
    else
    if (w = 0) and (x > -1) then
      y := i
    else
    if (w > 0) and (x > -1) then
    begin
      if (y-x > lastmax) or (lastmax = -1) then
      begin
        lastmax := y-x;
        lastx := x;
        lasty := y;
      end;
      x := -1;
      y := -1;
    end;
  end;

  if (w = 0) and (x > -1) then // last segment = 0
  begin
    if (y-x > lastmax) or (lastmax = -1) then
    begin
      lastmax := y-x;
      lastx := x;
      lasty := y;
    end
  end;

  b := False; // only one ::
  for i := 0 to 7 do
  begin
    if i in [lastx..lasty] then
    begin
      if not b then
      begin
        b := True;
        Result := Result + '::'
      end;
    end
    else
    begin
      if Length(Result) > 0 then
      begin
        if Result[Length(Result)] = ':' then
          Result := Result + IntToHex(GetWords(i), 0)
        else
          Result := Result + ':' + IntToHex(GetWords(i), 0)
      end
      else
        Result := Result + IntToHex(GetWords(i), 0);
    end
  end;
end;

function TIPv6.GetAsAlternativeString: string;
var
  i: Integer;
begin
  for i := 0 to 7 do
  begin
    Result := Result + IntToHex(GetWords(i), 0);
    if (i < 7) then
      Result := Result + ':'
  end;
end;

function TIPv6.GetWords(Index: Byte): Word;
begin
  if (0 <= Index) and (Index <= 7) then
    Result := (FRaw[2*Index] shl 8) or FRaw[2*Index+1]
  else
    Result := 0;
end;

procedure TIPv6.SetAsString(AValue: string);
var
  s: string;
  ts: string;
  c: Byte;
  i: Integer;
  checkIPv4: Boolean;
  ipv4: TIPv4;
begin
  Clear;
  ts := '';
  c := 0;
  s := IPv6Expand(Trim(AValue));
  checkIPv4 := False;
  if not IsValidIPv6Expanded(s) then
    Exit;
  for i := 1 to Length(s) do
  begin
    if (s[i] = ':') then
    begin
      SetWords(c, Hex2Dec(ts));
      Inc(c);
      ts := '';
    end
    else
    if s[i] in ['0'..'9', 'a'..'f', 'A'..'F'] then
    begin
      ts := ts + s[i];
    end
    else
    if (s[i] = '.') and (c = 6) then // IPv4 in last segment
    begin
      ts := ts + s[i];
      checkIPv4 := True;
    end
    else
    begin
      Exit;
    end;

    if (c > 7) or ((Length(ts) > 4) and (not checkIPv4)) then // segment length > 4
    begin
      Exit;
    end;
    //
    //if (c > 1) then
    //begin
    //  // :: shouldn't be here, first was deleted on expanding
    //  if (s[i-1] = s[i]) and (s[i] = ':') then
    //  begin
    //    Result := False;
    //    Exit;
    //  end;
    //end;
  end;

  if checkIPv4 then
  begin
    ipv4.AsString := ts;
    SetWords(6, ipv4.AsDWord shr 16);
    SetWords(7, ipv4.AsDWord and $ffff);
  end
  else
  begin
    SetWords(7, Hex2Dec(ts));
  end;
end;

procedure TIPv6.SetBits(Index: Byte; AValue: Byte);

  function change_bit(val, num: Byte; bitval: Boolean): Byte; inline;
  begin
    if bitval then
      Result := val or (1 shl num)
    else
      Result := val and (not (1 shl num))
  end;

var
  tmp: Byte;
  bitidx: Byte;
  arridx: Byte;
begin
  if (1 <= Index) and (Index <= 128) then
  begin
    arridx := (Index - 1) div 8;
    bitidx := 8 - (Index mod 8);
    if bitidx = 8 then
      bitidx := 0;
    FRaw[arridx] := change_bit(FRaw[arridx], bitidx, AValue > 0);
  end
end;

procedure TIPv6.SetBytes(Index: Byte; AValue: Byte);
begin
  if (0 <= Index) and (Index <= 15) then
    FRaw[Index] := AValue;
end;

procedure TIPv6.SetWords(Index: Byte; AValue: Word);
begin
  if (0 <= Index) and (Index <= 7) then
  begin
    FRaw[2*Index] := AValue shr 8;
    FRaw[2*Index+1] := AValue and $ff;
  end;
end;

function TIPv6.GetAsRawString: string;
var
  i: Integer;
begin
  Result := '';
  for i := 0 to 15 do
    Result := Result + IntToHex(FRaw[i], 2);
end;

procedure TIPv6.Clear;
var
  i: Byte;
begin
  for i := 0 to 15 do
    FRaw[i] := 0;
end;

class operator TIPv6.Equal(a, b: TIPv6): Boolean;
var
  i: Byte;
begin
  Result := True;
  for i := 0 to 15 do
    if a.FRaw[i] <> b.FRaw[i] then
    begin
      Result := False;
      Break;
    end;
end;

class operator TIPv6.NotEqual(a, b: TIPv6): Boolean;
begin
  Result := not (a = b);
end;

class operator TIPv6.GreaterThan(a, b: TIPv6): Boolean;
begin
  Result := a.GetAsRawString > b.GetAsRawString;
end;

class operator TIPv6.GreaterThanOrEqual(a, b: TIPv6): Boolean;
begin
  Result := a.GetAsRawString >= b.GetAsRawString;
end;

class operator TIPv6.LessThan(a, b: TIPv6): Boolean;
begin
  Result := a.GetAsRawString < b.GetAsRawString;
end;

class operator TIPv6.LessThanOrEqual(a, b: TIPv6): Boolean;
begin
  Result := a.GetAsRawString <= b.GetAsRawString;
end;

class operator TIPv6.Inc(a: TIPv6): TIPv6;

  procedure x(i: Byte);
  var
    d: Byte;
  begin
    if i >= 0 then
    begin
      d := a.FRaw[i];
      a.FRaw[i] := a.FRaw[i] + 1;
      if d = $ff then
        x(i-1);
    end;
  end;

begin
  x(15);
  Result := a;
end;

class operator TIPv6.Dec(a: TIPv6): TIPv6;

  procedure x(i: Byte);
  var
    d: Byte;
  begin
    if i >= 0 then
    begin
      d := a.FRaw[i];
      a.FRaw[i] := a.FRaw[i] - 1;
      if d = 0 then
        x(i-1);
    end;
  end;

begin
  x(15);
  Result := a;
end;

class operator TIPv6.BitwiseAnd(a, b: TIPv6): TIPv6;
var
  i: Byte;
begin
  Result := a;
  for i := 0 to 15 do
    Result.FRaw[i] := Result.FRaw[i] and b.FRaw[i];
end;

class operator TIPv6.BitwiseOr(a, b: TIPv6): TIPv6;
var
  i: Byte;
begin
  Result := a;
  for i := 0 to 15 do
    Result.FRaw[i] := Result.FRaw[i] or b.FRaw[i];
end;

class operator TIPv6.BitwiseXor(a, b: TIPv6): TIPv6;
var
  i: Byte;
begin
  Result := a;
  for i := 0 to 15 do
    Result.FRaw[i] := Result.FRaw[i] xor b.FRaw[i];
end;

class operator TIPv6.LogicalNot(a: TIPv6): TIPv6;
var
  i: Byte;
begin
  Result := a;
  for i := 0 to 15 do
    Result.FRaw[i] := not Result.FRaw[i];
end;

{ TIPv4 }

function TIPv4.GetAsString: string;
begin
  Result := IntToStr(GetOctet(0)) + '.' +
            IntToStr(GetOctet(1)) + '.' +
            IntToStr(GetOctet(2)) + '.' +
            IntToStr(GetOctet(3));
end;

function TIPv4.GetOctet(Index: Byte): Byte;
begin
  case Index of
    0: Result := (FRaw shr 24) and $ff;
    1: Result := (FRaw shr 16) and $ff;
    2: Result := (FRaw shr  8) and $ff;
    3: Result := (FRaw       ) and $ff;
  end;
end;

function TIPv4.GetAsBin: string;
begin
  Result := IntToBin(GetOctet(0), 8) + '.' +
            IntToBin(GetOctet(1), 8) + '.' +
            IntToBin(GetOctet(2), 8) + '.' +
            IntToBin(GetOctet(3), 8);
end;

function TIPv4.GetAsHex: string;
begin
  Result := IntToHex(GetOctet(0), 2) + '.' +
            IntToHex(GetOctet(1), 2) + '.' +
            IntToHex(GetOctet(2), 2) + '.' +
            IntToHex(GetOctet(3), 2);
end;

procedure TIPv4.SetAsString(AValue: string);
var
  i: Integer;
  tmp: string;
  b: Byte;
begin
  FRaw := 0;
  if not IsValidIPv4(AValue) then
    Exit;
  tmp := '';
  b := 24;
  for i := 1 to Length(AValue) do
  begin
    if AValue[i] = '.' then
    begin
      FRaw := FRaw or (StrToIntDef(tmp, 0) shl b);
      Dec(b, 8);
      tmp := '';
    end
    else
      tmp := tmp + AValue[i];
  end;
  FRaw := FRaw or StrToIntDef(tmp, 0);
end;

procedure TIPv4.Clear;
begin
  FRaw := 0;
end;

class operator TIPv4.Equal(a, b: TIPv4): Boolean;
begin
  Result := a.FRaw = b.FRaw;
end;

class operator TIPv4.NotEqual(a, b: TIPv4): Boolean;
begin
  Result := a.FRaw <> b.FRaw;
end;

class operator TIPv4.GreaterThan(a, b: TIPv4): Boolean;
begin
  Result := a.FRaw > b.FRaw;
end;

class operator TIPv4.GreaterThanOrEqual(a, b: TIPv4): Boolean;
begin
  Result := a.FRaw >= b.FRaw;
end;

class operator TIPv4.LessThan(a, b: TIPv4): Boolean;
begin
  Result := a.FRaw < b.FRaw;
end;

class operator TIPv4.LessThanOrEqual(a, b: TIPv4): Boolean;
begin
  Result := a.FRaw <= b.FRaw;
end;

class operator TIPv4.Inc(a: TIPv4): TIPv4;
begin
  Result.FRaw := a.FRaw + 1;
end;

class operator TIPv4.Dec(a: TIPv4): TIPv4;
begin
  Result.FRaw := a.FRaw - 1;
end;

class operator TIPv4.BitwiseAnd(a, b: TIPv4): TIPv4;
begin
  Result.FRaw := a.FRaw and b.FRaw;
end;

class operator TIPv4.BitwiseOr(a, b: TIPv4): TIPv4;
begin
  Result.FRaw := a.FRaw or b.FRaw;
end;

class operator TIPv4.BitwiseXor(a, b: TIPv4): TIPv4;
begin
  Result.FRaw := a.FRaw xor b.FRaw;
end;

class operator TIPv4.LogicalNot(a: TIPv4): TIPv4;
begin
  Result.FRaw := not a.FRaw;
end;

function IPv6Expand(s: string): string;
var
  i, l: Integer;
  c: Byte;
  x: Boolean;
  cs, ls, rs: string;
  pblyIPv4: Boolean;
begin
  pblyIPv4 := False;
  Result := '';
  ls := ''; // left
  cs := ''; // center
  rs := ''; // right
  c := 0;     // if : then c++
  x := False; // if :: then x = true
  l := Length(s);
  for i := 1 to l - 1 do
  begin
    if x then
      rs := rs + s[i]
    else
      ls := ls + s[i];

    if s[i] = ':' then
    begin
      if s[i] = s[i+1] then
        x := True;
      Inc(c);
    end;

    if s[i] = '.' then
      pblyIPv4 := True;
  end;

  if l > 0 then
  begin
    if x then
      rs := rs + s[l]
    else
      ls := ls + s[l];

    if (ls = ':') and x then // s = "::"  ls = ":"  rs = ":"  =>  ls = "0:"
      ls := '0' + ls;

    if (rs = ':') and x then // s = "::"  ls = ":"  rs = ":"  =>  rs = ":0"
    begin
      rs := rs + '0';
      Inc(c);
    end
  end;

  if x then // min 2x ":"
  begin
    if pblyIPv4 then
      l := 6
    else
      l := 7;
    for i := c to l do
    begin
      if Length(cs) = 0 then
        cs := '0'
      else
        cs := cs + ':0';
    end;
  end;

  if (Length(cs) = 0) and x then
    Result := Copy(ls, 1, Length(ls)-1) + rs // avoid "::" in ::2:3:4:5:6:7 => 0:<cs>:2:3:4:5:6:7
  else
    Result := ls + cs + rs;
end;

function IsValidIPv6(s: string): Boolean;
begin
  Result := IsValidIPv6Expanded(IPv6Expand(Trim(s)));
end;

function IPv6NetworkStart(AIP: TIPv6; AMask: Byte): TIPv6;
var
  i: Byte;
begin
  Result := AIP;
  for i := 1 to 128 do
    if i > AMask then
      Result.Bits[i] := 0;
end;

function IPv6NetworkEnd(AIP: TIPv6; AMask: Byte): TIPv6;
var
  i: Byte;
begin
  Result := AIP;
  for i := 1 to 128 do
    if i > AMask then
      Result.Bits[i] := 1;
end;

function IPv6ToStr(AValue: TIPv6): string;
begin
  Result := AValue.AsString;
end;

function StrToIPv6(AValue: string): TIPv6;
begin
  Result.AsString := AValue;
end;

function IPv6BlockType(AIP: TIPv6; AMask: Byte): TIPv6BlockType;

  function CheckBlock(AInIP: string; AMaskPrefix, AMaskBits: Integer): Boolean;
  var
    ip: TIPv6;
  begin
    ip.AsString := AInIP;
    Result := (IPv6NetworkStart(ip, mask) <= AIP) and (AIP <= IPv6NetworkEnd(ip, mask))
          and (AMaskBits >= AMaskPrefix);
  end;

var
  i: Byte;
begin
  if CheckBlock('::1', 128, AMask) then
    Result := i6btLoopback
  else
  if CheckBlock('fe80::', 10, AMask) then
    Result := i6btLinkLocal
  else
  if CheckBlock('fc00::', 7, AMask) then
    Result := i6btULA
  else
  if CheckBlock('::', 96, AMask) then
    Result := i6btIPv4Compatibe
  else
  if CheckBlock('::ffff:0:0', 96, AMask) then
    Result := i6btIPv4Mapped
  else
  if CheckBlock('::ffff:0:0:0', 96, AMask) then
    Result := i6btSIIT
  else
  if CheckBlock('2002::', 16, AMask) then
    Result := i6bt6to4
  else
  if CheckBlock('2001:10::', 28, AMask) then
    Result := i6btORCHID
  else
  if CheckBlock('2001:2::', 48, AMask) then
    Result := i6btBMWG
  else
  if CheckBlock('2001::', 32, AMask) then
    Result := i6btTeredo
  else
  if CheckBlock('2001:db8::', 32, AMask) then
    Result := i6btDocumentation
  else
  if CheckBlock('ff00::0', 8, AMask) then
    Result := i6btMulticast
  else
    Result := i6btNormal;
end;

function IPv6BlockTypeToStr(ABlockType: TIPv6BlockType): string;
begin
  case ABlockType of
    i6btNormal: Result := RS_IP6_NORMAL;
    i6btLoopback: Result := RS_IP6_LOOPBACK;
    i6btLinkLocal: Result := RS_IP6_LINK_LOCAL;
    i6btIPv4Compatibe: Result := RS_IP6_IPv4;
    i6btULA: Result := RS_IP6_ULA;
    i6btSIIT: Result := RS_IP6_SIIT;
    i6bt6to4: Result := RS_IP6_6_TO_4;
    i6btTeredo: Result := RS_IP6_TEREDO;
    i6btORCHID: Result := RS_IP6_ORCHID;
    i6btBMWG: Result := RS_IP6_BMWG;
    i6btDocumentation: Result := RS_IP6_DOCUMENTATION;
    i6btMulticast: Result := RS_IP6_MULTICAST;
  end
end;

function IPv4toIPv6Compatible(host: TIPv4): TIPv6;
begin
  Result.Clear;
  Result.Words[6] := host.AsDWord shr 16;
  Result.Words[7] := host.AsDWord and $ff;
end;

function IPv4toIPv6Mapped(host: TIPv4): TIPv6;
begin
  Result.Clear;
  Result.Words[5] := $ffff;
  Result.Words[6] := host.AsDWord shr 16;
  Result.Words[7] := host.AsDWord and $ff;
end;

function IPv4toIPv66to4(host: TIPv4): TIPv6;
begin
  // http://en.wikipedia.org/wiki/6to4
  Result.Clear;
  Result.Words[0] := $2002;
  Result.Words[1] := host.AsDWord shr 16;
  Result.Words[2] := host.AsDWord and $ff;
end;

function IPv4toIPv66over4(host: TIPv4): TIPv6;
begin
  Result.Clear;
  Result.Words[0] := $fe80;
  Result.Words[6] := host.AsDWord shr 16;
  Result.Words[7] := host.AsDWord and $ff;
end;

function IPv4toIPv6Teredo(host, server: TIPv4; port: Word): TIPv6;
var
  dx: DWord;
begin
  Result.Clear;
  Result.Words[0] := $2001;
  Result.Words[1] := $0000;
  Result.Words[2] := server.AsDWord shr 16;
  Result.Words[3] := server.AsDWord and $ffff;
  Result.Words[4] := $8000;
  Result.Words[5] := port xor $ffff;
  dx := host.AsDWord;
  dx := dx xor $ffffffff;
  Result.Words[6] := dx shr 16;
  Result.Words[7] := dx and $ffff;
end;

function IPv4toIPv6SIIT(host: TIPv4): TIPv6;
begin
  // http://en.wikipedia.org/wiki/IPv6_transition_mechanisms#Stateless_IP.2FICMP_Translation_.28SIIT.29
  Result.Clear;
  Result.Words[4] := $ffff;
  Result.Words[5] := $0000;
  Result.Words[6] := host.AsDWord shr 16;
  Result.Words[7] := host.AsDWord and $ff;
end;

{
https://bitbucket.org/intermapper/ipv6-validator

function IPv6SelfCheck: Boolean;

  procedure XTEST(a: Boolean; b, c: string);
  begin
    if a <> IsValidIPv6(b) then
    begin
      raise Exception.Create(b);
    end;
  end;

begin
  XTEST(false,'','---');
  XTEST(true ,'2001:0000:1234:0000:0000:C1C0:ABCD:0876','2001:0:1234::C1C0:ABCD:876');
  XTEST(true ,'3ffe:0b00:0000:0000:0001:0000:0000:000a','3ffe:b00::1:0:0:a');
  XTEST(true ,'FF02:0000:0000:0000:0000:0000:0000:0001','FF02::1');
  XTEST(true ,'0000:0000:0000:0000:0000:0000:0000:0001','::1');
  XTEST(true ,'0000:0000:0000:0000:0000:0000:0000:0000','::');
  XTEST(true ,'::ffff:192.168.1.26','::ffff:192.168.1.26');
  XTEST(false,'02001:0000:1234:0000:0000:C1C0:ABCD:0876','---');
  XTEST(false,'2001:0000:1234:0000:00001:C1C0:ABCD:0876','---');
  XTEST(true ,' 2001:0000:1234:0000:0000:C1C0:ABCD:0876',' 2001:0:1234::C1C0:ABCD:876');
  XTEST(true ,' 2001:0000:1234:0000:0000:C1C0:ABCD:0876  ',' 2001:0:1234::C1C0:ABCD:876  ');
  XTEST(false,' 2001:0000:1234:0000:0000:C1C0:ABCD:0876  0','---');
  XTEST(false,'2001:0000:1234: 0000:0000:C1C0:ABCD:0876','---');
  XTEST(false,'2001:1:1:1:1:1:255Z255X255Y255','---');

  //??XTEST(false,'3ffe:0b00:0000:0001:0000:0000:000a','---');
  XTEST(false,'FF02:0000:0000:0000:0000:0000:0000:0000:0001','---');
  XTEST(false,'3ffe:b00::1::a','---');
  XTEST(false,'::1111:2222:3333:4444:5555:6666::','---');
  XTEST(true ,'2::10','2::10');
  XTEST(true ,'ff02::1','ff02::1');
  XTEST(true ,'fe80::','fe80::');
  XTEST(true ,'2002::','2002::');
  XTEST(true ,'2001:db8::','2001:db8::');
  XTEST(true ,'2001:0db8:1234::','2001:db8:1234::');
  XTEST(true ,'::ffff:0:0','::ffff:0:0');
  XTEST(true ,'::1','::1');
  XTEST(true ,'::ffff:192.168.1.1','::ffff:192.168.1.1');
  XTEST(true ,'1:2:3:4:5:6:7:8','1:2:3:4:5:6:7:8');
  XTEST(true ,'1:2:3:4:5:6::8','1:2:3:4:5:6:0:8');
  XTEST(true ,'1:2:3:4:5::8','1:2:3:4:5::8');
  XTEST(true ,'1:2:3:4::8','1:2:3:4::8');
  XTEST(true ,'1:2:3::8','1:2:3::8');
  XTEST(true ,'1:2::8','1:2::8');
  XTEST(true ,'1::8','1::8');
  XTEST(true ,'1::2:3:4:5:6:7','1:0:2:3:4:5:6:7');
  XTEST(true ,'1::2:3:4:5:6','1::2:3:4:5:6');
  XTEST(true ,'1::2:3:4:5','1::2:3:4:5');
  XTEST(true ,'1::2:3:4','1::2:3:4');
  XTEST(true ,'1::2:3','1::2:3');
  XTEST(true ,'1::8','1::8');
  XTEST(true ,'::2:3:4:5:6:7:8','0:2:3:4:5:6:7:8');
  XTEST(true ,'::2:3:4:5:6:7','::2:3:4:5:6:7');
  XTEST(true ,'::2:3:4:5:6','::2:3:4:5:6');
  XTEST(true ,'::2:3:4:5','::2:3:4:5');
  XTEST(true ,'::2:3:4','::2:3:4');
  XTEST(true ,'::2:3','::2:3');
  XTEST(true ,'::8','::8');
  XTEST(true ,'1:2:3:4:5:6::','1:2:3:4:5:6::');
  XTEST(true ,'1:2:3:4:5::','1:2:3:4:5::');
  XTEST(true ,'1:2:3:4::','1:2:3:4::');
  XTEST(true ,'1:2:3::','1:2:3::');
  XTEST(true ,'1:2::','1:2::');
  XTEST(true ,'1::','1::');
  XTEST(true ,'1:2:3:4:5::7:8','1:2:3:4:5:0:7:8');
  XTEST(false,'1:2:3::4:5::7:8','---');
  XTEST(false,'12345::6:7:8','---');
  XTEST(true ,'1:2:3:4::7:8','1:2:3:4::7:8');
  XTEST(true ,'1:2:3::7:8','1:2:3::7:8');
  XTEST(true ,'1:2::7:8','1:2::7:8');
  XTEST(true ,'1::7:8','1::7:8');
  XTEST(true ,'1:2:3:4:5:6:1.2.3.4','1:2:3:4:5:6:1.2.3.4');
  XTEST(true ,'1:2:3:4:5::1.2.3.4','1:2:3:4:5:0:1.2.3.4');
  XTEST(true ,'1:2:3:4::1.2.3.4','1:2:3:4::1.2.3.4');
  XTEST(true ,'1:2:3::1.2.3.4','1:2:3::1.2.3.4');
  XTEST(true ,'1:2::1.2.3.4','1:2::1.2.3.4');
  XTEST(true ,'1::1.2.3.4','1::1.2.3.4');
  XTEST(true ,'1:2:3:4::5:1.2.3.4','1:2:3:4:0:5:1.2.3.4');
  XTEST(true ,'1:2:3::5:1.2.3.4','1:2:3::5:1.2.3.4');
  XTEST(true ,'1:2::5:1.2.3.4','1:2::5:1.2.3.4');
  XTEST(true ,'1::5:1.2.3.4','1::5:1.2.3.4');
  XTEST(true ,'1::5:11.22.33.44','1::5:11.22.33.44');
  XTEST(false,'1::5:400.2.3.4','---');
  XTEST(false,'1::5:260.2.3.4','---');
  XTEST(false,'1::5:256.2.3.4','---');
  XTEST(false,'1::5:1.256.3.4','---');
  XTEST(false,'1::5:1.2.256.4','---');
  XTEST(false,'1::5:1.2.3.256','---');
  XTEST(false,'1::5:300.2.3.4','---');
  XTEST(false,'1::5:1.300.3.4','---');
  XTEST(false,'1::5:1.2.300.4','---');
  XTEST(false,'1::5:1.2.3.300','---');
  XTEST(false,'1::5:900.2.3.4','---');
  XTEST(false,'1::5:1.900.3.4','---');
  XTEST(false,'1::5:1.2.900.4','---');
  XTEST(false,'1::5:1.2.3.900','---');
  XTEST(false,'1::5:300.300.300.300','---');
  XTEST(false,'1::5:3000.30.30.30','---');
  XTEST(false,'1::400.2.3.4','---');
  XTEST(false,'1::260.2.3.4','---');
  XTEST(false,'1::256.2.3.4','---');
  XTEST(false,'1::1.256.3.4','---');
  XTEST(false,'1::1.2.256.4','---');
  XTEST(false,'1::1.2.3.256','---');
  XTEST(false,'1::300.2.3.4','---');
  XTEST(false,'1::1.300.3.4','---');
  XTEST(false,'1::1.2.300.4','---');
  XTEST(false,'1::1.2.3.300','---');
  XTEST(false,'1::900.2.3.4','---');
  XTEST(false,'1::1.900.3.4','---');
  XTEST(false,'1::1.2.900.4','---');
  XTEST(false,'1::1.2.3.900','---');
  XTEST(false,'1::300.300.300.300','---');
  XTEST(false,'1::3000.30.30.30','---');
  XTEST(false,'::400.2.3.4','---');
  XTEST(false,'::260.2.3.4','---');
  XTEST(false,'::256.2.3.4','---');
  XTEST(false,'::1.256.3.4','---');
  XTEST(false,'::1.2.256.4','---');
  XTEST(false,'::1.2.3.256','---');
  XTEST(false,'::300.2.3.4','---');
  XTEST(false,'::1.300.3.4','---');
  XTEST(false,'::1.2.300.4','---');
  XTEST(false,'::1.2.3.300','---');
  XTEST(false,'::900.2.3.4','---');
  XTEST(false,'::1.900.3.4','---');
  XTEST(false,'::1.2.900.4','---');
  XTEST(false,'::1.2.3.900','---');
  XTEST(false,'::300.300.300.300','---');
  XTEST(false,'::3000.30.30.30','---');
  XTEST(true ,'fe80::217:f2ff:254.7.237.98','fe80::217:f2ff:254.7.237.98');
  XTEST(true ,'fe80::217:f2ff:fe07:ed62','fe80::217:f2ff:fe07:ed62');
  XTEST(true ,'2001:DB8:0:0:8:800:200C:417A','2001:DB8::8:800:200C:417A');
  XTEST(true ,'FF01:0:0:0:0:0:0:101','FF01::101');
  XTEST(true ,'0:0:0:0:0:0:0:1','::1');
  XTEST(true ,'0:0:0:0:0:0:0:0','::');
  XTEST(true ,'2001:DB8::8:800:200C:417A','2001:DB8::8:800:200C:417A');
  XTEST(true ,'FF01::101','FF01::101');
  XTEST(true ,'::1','::1');
  XTEST(true ,'::','::');
  XTEST(true ,'0:0:0:0:0:0:13.1.68.3','::13.1.68.3');
  XTEST(true ,'0:0:0:0:0:FFFF:129.144.52.38','::FFFF:129.144.52.38');
  XTEST(true ,'::13.1.68.3','::13.1.68.3');
  XTEST(true ,'::FFFF:129.144.52.38','::FFFF:129.144.52.38');
// 	# XTEST(true ,'2001:0DB8:0000:CD30:0000:0000:0000:0000/60','2001:0DB8:0000:CD30:0000:0000:0000:0000/60');
// 	# XTEST(true ,'2001:0DB8::CD30:0:0:0:0/60','2001:0DB8::CD30:0:0:0:0/60');
// 	# XTEST(true ,'2001:0DB8:0:CD30::/60','2001:0DB8:0:CD30::/60');
// 	# XTEST(true ,'::/128','::/128');
// 	# XTEST(true ,'::1/128','::1/128');
// 	# XTEST(true ,'FF00::/8','FF00::/8');
// 	# XTEST(true ,'FE80::/10','FE80::/10');
// 	# XTEST(true ,'FEC0::/10','FEC0::/10');
// 	# XTEST(false,'124.15.6.89/60','---');
  XTEST(false,'2001:DB8:0:0:8:800:200C:417A:221','---');
  XTEST(false,'FF01::101::2','---');
  XTEST(false,'','---');

  XTEST(true ,'fe80:0000:0000:0000:0204:61ff:fe9d:f156','fe80::204:61ff:fe9d:f156');
  XTEST(true ,'fe80:0:0:0:204:61ff:fe9d:f156','fe80::204:61ff:fe9d:f156');
  XTEST(true ,'fe80::204:61ff:fe9d:f156','fe80::204:61ff:fe9d:f156');
  //??XTEST(false,'fe80:0000:0000:0000:0204:61ff:254.157.241.086','---');
  XTEST(true ,'fe80:0:0:0:204:61ff:254.157.241.86','fe80::204:61ff:254.157.241.86');
  XTEST(true ,'fe80::204:61ff:254.157.241.86','fe80::204:61ff:254.157.241.86');
  XTEST(true ,'::1','::1');
  XTEST(true ,'fe80::','fe80::');
  XTEST(true ,'fe80::1','fe80::1');
  XTEST(false,':','---');

// Aeron supplied these test cases.
  XTEST(false,'1111:2222:3333:4444::5555:','---');
  XTEST(false,'1111:2222:3333::5555:','---');
  XTEST(false,'1111:2222::5555:','---');
  XTEST(false,'1111::5555:','---');
  XTEST(false,'::5555:','---');
  XTEST(false,':::','---');
  XTEST(false,'1111:','---');
  XTEST(false,':','---');

  XTEST(false,':1111:2222:3333:4444::5555','---');
  XTEST(false,':1111:2222:3333::5555','---');
  XTEST(false,':1111:2222::5555','---');
  XTEST(false,':1111::5555','---');
  XTEST(false,':::5555','---');
  XTEST(false,':::','---');

  XTEST(false,'1.2.3.4:1111:2222:3333:4444::5555','---');
  XTEST(false,'1.2.3.4:1111:2222:3333::5555','---');
  XTEST(false,'1.2.3.4:1111:2222::5555','---');
  XTEST(false,'1.2.3.4:1111::5555','---');
  XTEST(false,'1.2.3.4::5555','---');
  XTEST(false,'1.2.3.4::','---');

// Additional Patterns
// from http://rt.cpan.org/Public/Bug/Display.html?id=50693

  XTEST(true ,'2001:0db8:85a3:0000:0000:8a2e:0370:7334','2001:db8:85a3::8a2e:370:7334');
  XTEST(true ,'2001:db8:85a3:0:0:8a2e:370:7334','2001:db8:85a3::8a2e:370:7334');
  XTEST(true ,'2001:db8:85a3::8a2e:370:7334','2001:db8:85a3::8a2e:370:7334');
  XTEST(true ,'2001:0db8:0000:0000:0000:0000:1428:57ab','2001:db8::1428:57ab');
  XTEST(true ,'2001:0db8:0000:0000:0000::1428:57ab','2001:db8::1428:57ab');
  XTEST(true ,'2001:0db8:0:0:0:0:1428:57ab','2001:db8::1428:57ab');
  XTEST(true ,'2001:0db8:0:0::1428:57ab','2001:db8::1428:57ab');
  XTEST(true ,'2001:0db8::1428:57ab','2001:db8::1428:57ab');
  XTEST(true ,'2001:db8::1428:57ab','2001:db8::1428:57ab');
  XTEST(true ,'0000:0000:0000:0000:0000:0000:0000:0001','::1');
  XTEST(true ,'::1','::1');
  XTEST(true ,'::ffff:12.34.56.78','::ffff:12.34.56.78');
  XTEST(true ,'::ffff:0c22:384e','::ffff:c22:384e');
  XTEST(true ,'2001:0db8:1234:0000:0000:0000:0000:0000','2001:db8:1234::');
  XTEST(true ,'2001:0db8:1234:ffff:ffff:ffff:ffff:ffff','2001:db8:1234:ffff:ffff:ffff:ffff:ffff');
  XTEST(true ,'2001:db8:a::123','2001:db8:a::123');
  XTEST(true ,'fe80::','fe80::');
  XTEST(true ,'::ffff:192.0.2.128','::ffff:192.0.2.128');
  XTEST(true ,'::ffff:c000:280','::ffff:c000:280');

  XTEST(false,'123','---');
  XTEST(false,'ldkfj','---');
  XTEST(false,'2001::FFD3::57ab','---');
  XTEST(false,'2001:db8:85a3::8a2e:37023:7334','---');
  XTEST(false,'2001:db8:85a3::8a2e:370k:7334','---');
  XTEST(false,'1:2:3:4:5:6:7:8:9','---');
  XTEST(false,'1::2::3','---');
  XTEST(false,'1:::3:4:5','---');
  XTEST(false,'1:2:3::4:5:6:7:8:9','---');
  XTEST(false,'::ffff:2.3.4','---');
  XTEST(false,'::ffff:257.1.2.3','---');
  XTEST(false,'1.2.3.4','---');

// Test collapsing zeroes...

  XTEST(true ,'a:b:c:d:e:f:f1:f2','a:b:c:d:e:f:f1:f2');
  XTEST(true ,'a:b:c::d:e:f:f1','a:b:c:0:d:e:f:f1');
  XTEST(true ,'a:b:c::d:e:f','a:b:c::d:e:f');
  XTEST(true ,'a:b:c::d:e','a:b:c::d:e');
  XTEST(true ,'a:b:c::d','a:b:c::d');
  XTEST(true ,'::a','::a');
  XTEST(true ,'::a:b:c','::a:b:c');
  XTEST(true ,'::a:b:c:d:e:f:f1','0:a:b:c:d:e:f:f1');
  XTEST(true ,'a::','a::');
  XTEST(true ,'a:b:c::','a:b:c::');
  XTEST(true ,'a:b:c:d:e:f:f1::','a:b:c:d:e:f:f1:0');
  XTEST(true ,'a:bb:ccc:dddd:000e:00f:0f::','a:bb:ccc:dddd:e:f:f:0');
  XTEST(true ,'0:a:0:a:0:0:0:a','0:a:0:a::a');
  XTEST(true ,'0:a:0:0:a:0:0:a','0:a::a:0:0:a');
  XTEST(true ,'2001:db8:1:1:1:1:0:0','2001:db8:1:1:1:1::');
  XTEST(true ,'2001:db8:1:1:1:0:0:0','2001:db8:1:1:1::');
  XTEST(true ,'2001:db8:1:1:0:0:0:0','2001:db8:1:1::');
  XTEST(true ,'2001:db8:1:0:0:0:0:0','2001:db8:1::');
  XTEST(true ,'2001:db8:0:0:0:0:0:0','2001:db8::');
  XTEST(true ,'2001:0:0:0:0:0:0:0','2001::');

  XTEST(true ,'A:BB:CCC:DDDD:000E:00F:0F::','A:BB:CCC:DDDD:E:F:F:0');

  XTEST(true ,'0:0:0:0:0:0:0:0','::');
  XTEST(true ,'0:0:0:0:0:0:0:a','::a');
  XTEST(true ,'0:0:0:0:a:0:0:0','::a:0:0:0');
  XTEST(true ,'0:0:0:a:0:0:0:0','0:0:0:a::');
  XTEST(true ,'0:a:0:0:a:0:0:a','0:a::a:0:0:a');
  XTEST(true ,'a:0:0:a:0:0:a:a','a::a:0:0:a:a');
  XTEST(true ,'a:0:0:a:0:0:0:a','a:0:0:a::a');
  XTEST(true ,'a:0:0:0:a:0:0:a','a::a:0:0:a');
  XTEST(true ,'a:0:0:0:a:0:0:0','a::a:0:0:0');
  XTEST(true ,'a:0:0:0:0:0:0:0','a::');
end;
}
end.
