unit ncipv4analyzer;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, FileUtil, Forms, Controls, Graphics, Dialogs, StdCtrls,
  ExtCtrls, EditBtn, ncstruct;

type

  { TfncIPv4Analyzer }

  TfncIPv4Analyzer = class(TFrame)
    edCIDR: TEdit;
    edIP: TEdit;
    edOSubnets: TEditButton;
    edMask: TEdit;
    edOBroadcast: TEdit;
    edOHostMax: TEdit;
    edOHostMin: TEdit;
    edOIP: TEdit;
    edOMask: TEdit;
    edONetwork: TEdit;
    edOPrivate: TLabel;
    edOPrivate1: TLabel;
    edOHosts: TEditButton;
    lbIIP: TLabel;
    lbIMask: TLabel;
    lbOIPAddress: TLabel;
    lbOMask: TLabel;
    lbONetwork: TLabel;
    lbOBroadcast: TLabel;
    lbOHN: TLabel;
    lbOFirstHost: TLabel;
    lbOLastHost: TLabel;
    RadioGroup1: TRadioGroup;
    procedure edIPChange(Sender: TObject);
    procedure edOSubnetsButtonClick(Sender: TObject);
  private
    FOnSubnetClick: TNotifyEvent;
    FUpdating: Boolean;
    function FormatInt(AIn: Int64): string;
    function FormatIPv4(AIn: TIPv4): string;
  public
    constructor Create(AOwner: TComponent); override;
    property OnSubnetClick: TNotifyEvent read FOnSubnetClick write FOnSubnetClick;
  end;

var
  fncIPv4Calc: TfncIPv4Analyzer;

implementation

{$R *.lfm}

uses
  StrUtils, nclang;

{ TfncIPv4Analyzer }

procedure TfncIPv4Analyzer.edIPChange(Sender: TObject);
var
  COLORS: array[Boolean] of TColor = (clRed, clBlack);
  AIsValid: Boolean;
  i: Int64;
  ip, mask: TIPv4;
begin
  // validation and mask conversion
  if (Sender = edIP) then
    AIsValid := IsValidIPv4(edIP.Text)
  else
  if (Sender = edMask) then
  begin
    AIsValid := IsValidMask(edMask.Text);
    if not FUpdating then
    begin
      FUpdating := True;
      if AIsValid then
        edCIDR.Text := IntToStr(MaskToCIDR(StrToIPv4(edMask.Text)))
      else
        edCIDR.Text := '0';
      FUpdating := False;
    end;
  end
  else
  if (Sender = edCIDR) then
  begin
    i := StrToIntDef(edCIDR.Text, -1);
    AIsValid := i in [0..32];
    if not FUpdating then
    begin
      FUpdating := True;
      if AIsValid then
        edMask.Text := CIDRToMask(i).AsString
      else
        edMask.Text := CIDRToMask(0).AsString;
      FUpdating := False;
    end
  end
  else
    AIsValid := True;

  if Sender is TEdit then
    TEdit(Sender).Font.Color := COLORS[AIsValid];

  // valid output
  if AIsValid then
  begin
    // init local var
    ip.AsString := edIP.Text;
    mask.AsString := edMask.Text;
    // ip info
    edOIP.Text := FormatIPv4(ip);
    edOMask.Text := FormatIPv4(mask);
    edONetwork.Text := FormatIPv4(IPv4Network(ip, mask));
    edOBroadcast.Text := FormatIPv4(IPv4Broadcast(ip, mask));
    // hosts
    i := IPv4Hosts(mask);
    case i of
      -1: edOHosts.Text := RS_SINGLE_HOST;
      -2: edOHosts.Text := RS_POINT_TO_POINT;
    else
      begin
        edOHosts.Text := FormatInt(i);
        edOHostMin.Text := FormatIPv4(IPv4FirstHost(ip, mask));
        edOHostMax.Text := FormatIPv4(IPv4LastHost(ip, mask));
      end;
    end;
    if (i = -2) or (i = -1) then
    begin
      edOHostMin.Text := '-';
      edOHostMax.Text := '-';
    end;
    edOSubnets.Text := FormatInt(IPv4Subnets(mask));
    // type & class
    edOPrivate.Caption := IPv4BlockTypeToStr(IPv4BlockType(ip, mask));
    if Length(edOPrivate.Caption) > 0 then
      edOPrivate.Caption := edOPrivate.Caption + ' / ';
    edOPrivate.Caption := edOPrivate.Caption + RS_CLASS + IPv4Class(ip);
  end
  else
  // error output
  begin
    edOIP.Text := RS_INVALID_INPUT;
    edOMask.Text := RS_INVALID_INPUT;
    edONetwork.Text := RS_INVALID_INPUT;
    edOBroadcast.Text := RS_INVALID_INPUT;
    edOHosts.Text := RS_INVALID_INPUT;
    edOSubnets.Text := RS_INVALID_INPUT;
    edOHostMin.Text := RS_INVALID_INPUT;
    edOHostMax.Text := RS_INVALID_INPUT;
    edOPrivate.Caption := '';
  end;
end;

procedure TfncIPv4Analyzer.edOSubnetsButtonClick(Sender: TObject);
begin
  if Assigned(FOnSubnetClick) then
    FOnSubnetClick(Sender);
end;

function TfncIPv4Analyzer.FormatIPv4(AIn: TIPv4): string;
begin
  case RadioGroup1.ItemIndex of
    0: Result := AIn.AsString;
    1: Result := IntToStr(AIn.AsDWord);
    2: Result := AIn.AsHex;
    3: Result := AIn.AsBin;
  end;
end;

constructor TfncIPv4Analyzer.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  edOSubnets.Glyph.Assign(edOHosts.Glyph);

  edIP.Text := NS_DEFAULT_EMPTY_IPv4;
  edMask.Text := NS_DEFAULT_EMPTY_IPv4;
  edIPChange(nil);
end;

function TfncIPv4Analyzer.FormatInt(AIn: Int64): string;
begin
  case RadioGroup1.ItemIndex of
    0: Result := IntToStr(AIn);
    1: Result := IntToStr(AIn);
    2: Result := IntToHex(AIn, 0);
    3: Result := IntToBin(AIn, 32, 8);
  end;
end;

end.

